/************************************************************************************************
 * Twilight : this class contains the MOON physics : the passive igloo project
 * Twilight 2016 is Copyright (c) 2016, Peter Gallinelli
 * 
 * created : May 2016 by peter gallinelli
 * history : 
 ************************************************************************************************/

import static java.lang.Math.*;

public class moonCalc {

    double DEG_IN_RADIAN = 180 / Math.PI;
    double HRS_IN_RADIAN = 12 / Math.PI;

    double raSun = 0; // (hrs)
    double decSun = 0; // declination (deg)
    double raMoon = 0; // (hrs)
    double decMoon = 1; // declination (deg)
    
    int[] JM = new int[14]; // cumulated days of month [d]
    
    void setJM(int[] J) {
        for (int i = 0; i < J.length; i++)
            JM[i] = J[i];
    }    
    
    /**
     * calculate moon azimut (deg from north) and altitude (deg from horizontal for the given date, time and location<br>
     * NOTE: longitude is negative for West
     * @param lat double : latitude (decimal degrees)
     * @param lon double : longitude (decimal degrees)
     * @param year double : date year
     * @param month double : date month (January = 1)
     * @param day double : date day (1 to 31)
     * @param hours double : time hour (0 to 23)
     * @param minutes double : time minute (0 to 59)
     * @param seconds double : time second (0 to 59)
     * @param timezone double : time zone hours relative to GMT/UTC (hours W- E+)
     * @param dlstime double : daylight savings time (0 = no, 1 = yes) (hours)
     * @param refractionOn boolean : true = YES, false = NO
     * @return double[] : [0] moon azimut in degrees from north<br>
     *                    [1] moon elevation in degrees from horizon
     *                    [2] moon shine (0 = new to 1 = full)
     *                    [3] moon phase (0 = new to 12 = full to 24))
     */
    public double[] calcMoonPosition(double lat, double lon, double year, double month, double day, double hours,
                                    double minutes, double seconds, double timezone, double dlstime,
                                    boolean refractionOn) {
        double longit = lon2longit(lon);
        double hoy = (JM[(int) month - 1] + day - 1) * 24 + hours + minutes / 60 + seconds / 3600;
        
        // calculate body elements
        double jd = hoy2julian((int) year, hoy, timezone, dlstime);
        lpsun(jd);
        double hSun = altit(decSun, lst(jd, longit) - raSun, lat);
        lpmoon(jd, lat, lst(jd, longit));
        double hMoon = altit(decMoon, lst(jd, longit) - raMoon, lat);
        double azMoon = 0; // TODO

        // calculate moon phase
        double shine = 0;
        if (raSun >= raMoon)
            shine = raSun - raMoon;
        else
            shine = raMoon - raSun;
        if (shine > 12)
            shine = 12 - (shine - 12);
        shine = shine / 12;

        // calculate moon phase 0...24
        double phase = raMoon - raSun;
        if (phase < 0)
            phase = phase + 24;
        
        /*if (phase < 12)
            shine = 1 - (12 - phase) / 12;
        else
            shine = (24 - phase) / 12;*/
        
        // return results : moon azimut, altitude, phase
        double[] res = {azMoon, hMoon, shine, phase};
        return res;
    }
    
    /**
     * just for testing
     */
    void testIt() {
        // user settings
        double lat = 77.5; // latitude (deg N+ S-)
        double lon = -67; // longitude (deg W- E+)
        int year = 2016; // year
        int zone = -4; // time zone (hrs W- E+)
        int dls = 0; // daylight savings (no=0 yes = 1)

        double jd;
        double dr = 0;
        double lastdr = 2;
        double longit = lon2longit(lon);

        System.out.println("Moon    year:m:d:hh  Sun/Moon height  Sun/Moon angle  moon phase");
        System.out.println("----------------------------------------------------------------");

        for (double hoy = 0; hoy < 1000; hoy = hoy + 1) {
            // calculate body elements
            jd = hoy2julian(year, hoy, zone, dls);
            lpsun(jd);
            double hSun = altit(decSun, lst(jd, longit) - raSun, lat);
            lpmoon(jd, lat, lst(jd, longit));
            double hMoon = altit(decMoon, lst(jd, longit) - raMoon, lat);

            // calculate moon phase
            if (raSun >= raMoon)
                dr = raSun - raMoon;
            else
                dr = raMoon - raSun;
            if (dr > 12)
                dr = 12 - (dr - 12);
            dr = dr / 12;
            String txt = "";
            for (int i = 0; i < 5; i++) {
                if (hMoon > 0 && i <= dr * 5)
                    txt = txt + "*";
                else
                    txt = txt + " ";
            }
            
            // print results
            String moon = "";
            if (lastdr == 2)
                moon = " "; // first cycle draw nothing
            else if(dr < 0.1)
                moon = "-"; // new moon
            else if (dr > 0.9)
                moon = "O"; // fullmoon
            else {
                if(dr > lastdr)
                    moon = ")"; // first quarter
                else
                    moon = "("; // last quarter
            }
            double[] res = hoy2date(hoy);
            String tx = "";
            if (res[2] < 10)
                tx = " ";
            System.out.print(moon + " ");
            System.out.print(txt + " " + year + ":" + (int) res[0] + ":" + (int) res[1] + ":" + tx + (int) res[2]);
            System.out.print("  h: " + String.format("%.2f", hSun) + "/" + String.format("%.2f", hMoon));
            System.out.print("  r: " + String.format("%.1f", raSun) + "/" + String.format("%.1f", raMoon));
            System.out.println("  p: " + String.format("%.2f", dr));
            lastdr = dr;
        }
        System.out.println("----------------------------------------------------------------");
        System.out.println("- new moon, ) first quarter, O fullmoon, ( last quarter");
        System.out.println("... done");
    }

    /**
     * Convert longitude (deg -W) to (dec hrs +W)
     * @param lon longitude (deg : E+, W-)
     * @return longitude (dec hrs : W+, E-)
     */
    double lon2longit(double lon) {
        double longit = (-lon - 180) * 12 / 180;
        return longit;
    }

    /**
     * Convert hour of year to Julian DATE
     * @param year double year
     * @param hoy double hour of year (decimal hour)
     * @param zone double time zone hours relative to GMT/UTC (hours)
     * @param dls double daylight savings time (0 = no, 1 = yes) (hours)
     * @return
     */
    double hoy2julian(int year, double hoy, double zone, double dls) {
        // extract month and day from hoy
        int doy = (int) Math.floor(hoy / 24);
        int m = 1;
        while (doy > JM[m])
            m++;
        double month = m;
        double day = (double) Math.max(1, doy - JM[m - 1] + 1);

        // calculate time
        double timenow = hoy - ((double) doy * 24.0) - zone - dls; // = hh + mm / 60 + ss / 3600 + zone;

        // get julian date
        double jd = (double) calcJD(year, month, day) + timenow / 24.0;
        return jd;
    }

    /**
     * Calculate Julian DAY from calendar day
     * @param year int : 4 digit year
     * @param month int : January = 1
     * @param day int : 1 - 31
     * @return double : julian day
     */
    int calcJD(double year, double month, double day) {
        if (month <= 2) {
            year = year - 1;
            month = month + 12;
        }
        double A = floor(year / 100);
        double B = 2 - A + floor(A / 4);
        double JD = floor(365.25 * (year + 4716)) + floor(30.6001 * (month + 1)) + day + B - 1524.5;
        return (int) JD;
    }

    /**
     * Date : transforms hour of year into month, day, hour aund minute
     * @param hoy double : hour of year [1...(yearDays * 24)]
     * @return double[] : [0] month, [1] day, [2] hour, [3] minute
     */
    public double[] hoy2date(double hoy) {
        int doy = (int) floor(hoy / 24);
        int hh = (int) hoy - doy * 24;
        int min = (int) ((hoy - floor(hoy)) * 60);
        int mm = 1;
        while (doy > JM[mm])
            mm++;
        int dd = doy - JM[mm - 1] + 1; // ------>>>  corriger dans twilight ! (+1)
        double[] result = { mm, dd, hh, min };
        return result;
    }

    /**
     * Create a Timestamp for the specified Julian date
     * @param jd : julian date
     * @return time in seconds from beginning of time (1970-01-01 00:00:00)
     */
    double julian2time_t(double jd) {
        return (((jd - 2440587.5) * 86400.0));
    }

    /**
     * Convert time t to Julian date
     * @param t : time in seconds from beginning of time (1970-01-01 00:00:00)
     * @return julian date
     */
    double time_t2julian(double t) {
        return ((t / 86400.0) + 2440587.5);
    }

    /**
     * Returns radian angle 0 to 2pi for coords x, y
     * @param x
     * @param y
     * @return
     */
    private double atan_circ(double x, double y) {
        double theta;
        if (x == 0) {
            if (y > 0.)
                theta = Math.PI / 2.;
            else if (y < 0.)
                theta = 3. * Math.PI / 2.;
            else
                theta = 0.; /* x and y zero */
        } else
            theta = Math.atan(y / x);
        if (x < 0.)
            theta = theta + Math.PI;
        if (theta < 0.)
            theta = theta + 2. * Math.PI;
        return theta;
    }

    /**
     * Returns altitude of body above horizon
     * @param dec body declination (deg)
     * @param ha body hour angle (hours)
     * @param lat observer latitude (degres)
     * @return altitude above horizon (deg)
     */
    double altit(double dec, double ha, double lat) {
        dec = dec / DEG_IN_RADIAN;
        ha = ha / HRS_IN_RADIAN;
        lat = lat / DEG_IN_RADIAN;
        double h =
            DEG_IN_RADIAN * Math.asin(Math.cos(dec) * Math.cos(ha) * Math.cos(lat) + Math.sin(dec) * Math.sin(lat));
        return h;
    }

    /**
     * Returns the local MEAN sidereal time (dec hrs) at julian date jd at west longitude long (decimal hours).
       Follows definitions in 1992 Astronomical Almanac, pp. B7 and L2.
       Expression for GMST at 0h UT referenced to Aoki et al, A&A 105, p.359, 1982.
     * @param jd Julian date
     * @param longit west longitude (dec hrs)
     * @return local MEAN sidereal time (dec hrs)
     */
    static double lst(double jd, double longit) {
        double t, ut, jdmid, jdint, jdfrac, sid_g;
        long jdin, sid_int;

        jdin = (long) jd; // fossil code from earlier package which split jd into integer and fractional parts ...
        jdint = jdin;
        jdfrac = jd - jdint;
        if (jdfrac < 0.5) {
            jdmid = jdint - 0.5;
            ut = jdfrac + 0.5;
        } else {
            jdmid = jdint + 0.5;
            ut = jdfrac - 0.5;
        }
        t = (jdmid - 2451545.0) / 36525; // jul cent. since J2000
        sid_g = (24110.54841 + 8640184.812866 * t + 0.093104 * t * t - 6.2e-6 * t * t * t) / 86400.0;
        sid_int = (long) sid_g;
        sid_g = sid_g - (double) sid_int;
        sid_g = sid_g + 1.0027379093 * ut - longit / 24.0;
        sid_int = (long) sid_g;
        sid_g = (sid_g - (double) sid_int) * 24.0;
        if (sid_g < 0.)
            sid_g = sid_g + 24.;
        return (sid_g);
    }


    /**
     * Calculates hour angle and declination of MOON
     * implements "low precision" moon algorithms from Astronomical Almanac (p. D46 in 1992 version).
     * Does apply the topocentric correction.
     * ra and dec are returned as decimal hours and decimal degrees
     * @param jd Julian date
     * @param lat latitude (deg)
     * @param sid sideral time (hrs)
     */
    void lpmoon(double jd, double lat, double sid) {
        double T, lambda, beta, pie, l, m, n, x, y, z, alpha, delta;
        double rad_lat, rad_lst, distance, topo_dist;

        T = (jd - 2451545.0) / 36525.0; // jul cent. since J2000

        lambda =
            218.32 + 481267.883 * T + 6.29 * sin((134.9 + 477198.85 * T) / DEG_IN_RADIAN) -
            1.27 * sin((259.2 - 413335.38 * T) / DEG_IN_RADIAN) + 0.66 * sin((235.7 + 890534.23 * T) / DEG_IN_RADIAN) +
            0.21 * sin((269.9 + 954397.70 * T) / DEG_IN_RADIAN) - 0.19 * sin((357.5 + 35999.05 * T) / DEG_IN_RADIAN) -
            0.11 * sin((186.6 + 966404.05 * T) / DEG_IN_RADIAN);
        lambda = lambda / DEG_IN_RADIAN;
        beta =
            5.13 * sin((93.3 + 483202.03 * T) / DEG_IN_RADIAN) + 0.28 * sin((228.2 + 960400.87 * T) / DEG_IN_RADIAN) -
            0.28 * sin((318.3 + 6003.18 * T) / DEG_IN_RADIAN) - 0.17 * sin((217.6 - 407332.20 * T) / DEG_IN_RADIAN);
        beta = beta / DEG_IN_RADIAN;
        pie =
            0.9508 + 0.0518 * cos((134.9 + 477198.85 * T) / DEG_IN_RADIAN) +
            0.0095 * cos((259.2 - 413335.38 * T) / DEG_IN_RADIAN) +
            0.0078 * cos((235.7 + 890534.23 * T) / DEG_IN_RADIAN) +
            0.0028 * cos((269.9 + 954397.70 * T) / DEG_IN_RADIAN);
        pie = pie / DEG_IN_RADIAN;
        distance = 1 / sin(pie);

        l = cos(beta) * cos(lambda);
        m = 0.9175 * cos(beta) * sin(lambda) - 0.3978 * sin(beta);
        n = 0.3978 * cos(beta) * sin(lambda) + 0.9175 * sin(beta);

        x = l * distance;
        y = m * distance;
        z = n * distance; // for topocentric correction
        // lat isn't passed right on some IBM systems unless you do this or something like it!
        rad_lat = lat / DEG_IN_RADIAN;
        rad_lst = sid / HRS_IN_RADIAN;
        x = x - cos(rad_lat) * cos(rad_lst);
        y = y - cos(rad_lat) * sin(rad_lst);
        z = z - sin(rad_lat);

        topo_dist = sqrt(x * x + y * y + z * z);

        l = x / topo_dist;
        m = y / topo_dist;
        n = z / topo_dist;

        alpha = atan_circ(l, m);
        delta = asin(n);
        raMoon = alpha * HRS_IN_RADIAN;
        decMoon = delta * DEG_IN_RADIAN;
    }


    /**
     * Calculates hour angle and declination of SUN
     * Low precision formulae for the sun, from Almanac p. C24 (1990)
     * ra and dec are returned as decimal hours and decimal degrees
     * @param jd Julian date
     */
    void lpsun(double jd) {
        double n, L, g, lambda, epsilon, x, y, z;

        n = jd - 2451545.0;
        L = 280.460 + 0.9856474 * n;
        g = (357.528 + 0.9856003 * n) / DEG_IN_RADIAN;
        lambda = (L + 1.915 * sin(g) + 0.020 * sin(2. * g)) / DEG_IN_RADIAN;
        epsilon = (23.439 - 0.0000004 * n) / DEG_IN_RADIAN;

        x = cos(lambda);
        y = cos(epsilon) * sin(lambda);
        z = sin(epsilon) * sin(lambda);

        raSun = (atan_circ(x, y)) * HRS_IN_RADIAN;
        decSun = (asin(z)) * DEG_IN_RADIAN;
    }

}





// ======================= licence ==========================


/*
    skycal.cc -- Functions for sunrise, sunset, phase of moon.
    Last modified 1998-01-25
    
    This source file began its life as skycalendar.c and skycalc.c in
    John Thorstensen's skycal distribution (version 4.1, 1994-09) at
    ftp://iraf.noao.edu/contrib/skycal.tar.Z.  Those portions that are
    unchanged from the original sources are covered by the original
    license statement, included below.  The new portions and "value
    added" by David Flater are covered under the GNU General Public
    License:

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// The original skycal comments and license statement are as follows:

/*
   This is a self-contained c-language program to print a nighttime
   astronomical calendar for use in planning observations.
   It prints to standard output (usually the terminal); the
   operator should capture this output (e. g., using redirection
   in UNIX or the /out= switch in VMS) and then print it on an
   appropriate output device.  The table which is printed is some
   125 columns wide, so a wide device is required (either a line
   printer or a laserprinter in LANDSCAPE mode.)  It is assumed that
   the ASCII form-feed character will actually begin a new page.
   The original program was to run on VMS, but it should be very
   transportable.  Non-vms users will probably want to change
   'unixio.h' to 'stdio.h' in the first line.
   An explanatory text is printed at the beginning of the output, which
   includes the appropriate CAUTIONS regarding accuracy and applicability.

   A number of 'canned site' parameters have been included.  Be
   careful of time zones, DST etc. for foreign sites.
   To customize to your own site, install an option in the
   routine 'load_site'.  The code is very straightforward; just do
   it exactly the same as the others.  You might also want to erase
   some seldom-used choices.  One can also specify new site parameters
   at run time.

   This program may be used freely by anyone for scientific or educational
   purposes.  If you use it for profit, I want a cut, and claim
   a copyright herewith.  In any case please acknowledge the source:

			John Thorstensen
			Dept. of Physics and Astronomy
			Dartmouth College
			Hanover, NH 03755
			John.Thorstensen@dartmouth.edu

			May 26, 1993.
*/